/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Kevin Cornell (Rational Software Corporation)
 *******************************************************************************/

/* Eclipse Program Launcher
 *
 * This program performs the launching of the eclipse program along
 * with the splash window. If the splash window is to be displayed,
 * the java application will receive two extra arguments:
 *     -showsplash  <splashCommand>
 *
 * When the Java program starts, it should determine if a different
 * version of the splash bitmaps are to be used (uses the feature mechanism).
 * If feature specific bitmaps are to be used, the root directory where
 * those bitmaps reside (e.g., <rootDir>/splash/<bitmapFile>) should be
 * appended (in double quotes) to the splash command. If a directory is
 * not appended, the Eclipse install directory is used. 
 * 
 * The Java program initiates the displaying of the splash window
 * by executing the splash command as follows:
 *
 *     String command = splashCommand;
 *     String directory = getFeatureRootDirectory();
 *     if (directory != null)
 *         command += " \"" + directory + "\"";
 *     Process splashProcess = Runtime.getRuntime().exec( command );
 *
 * When the Java program initialization is complete, the splash window
 * is brought down by destroying the splash process as follows:
 *
 *     splashProcess.destroy();
 *
 * Therefore, when the splash window is visible, there are actually three
 * processes running: 
 * 1) the main launcher process 
 * 2) the Java VM process (Eclipse) 
 * 3) the splash window process.
 *
 * The Java application does not need to know the format of the
 * <splashCommand> argument but the command format is:
 *   <fullPathOfLauncherProgram> -showsplash <timeout>
 *
 * In other words, another eclipse launcher program is run with
 * the arguments "-showsplash" and a timeout value for bringing
 * the splash window down.
 *
 * Similarly, the Java application will receive two other arguments:
 *    -exitdata <exitDataCommand>
 *
 * The <exitDataCommand> argument can be used by the Java application
 * to provide specific exit data to the main launcher process. It accomplishes
 * this by executing the exit data command with an extra argument, which is
 * the exit data.
 *
 *     String command = exitDataCommand;
 *     String data = "Some data";
 *     if (data != null)
 *         command += " \"" + data + "\"";
 *     Process exitDataProcess = Runtime.getRuntime().exec( command );
 *     exitDataProcess.waitFor();
 *
 * The exit data size must not exceed MAX_SHARED_LENGTH which is
 * 16Kb. The exit data process will exit with an exit code
 * different than 0 if that happens.
 *
 * The Java application does not need to know the format of the
 * <exitDataCommand> argument either, but the command format is:
 *   <fullPathOfLauncherProgram> -exitdata <id>
 *
 *   where id is a process independent identifier for the shared
 * memory segment that stores the exit data.
 *
 * The main launcher recognizes the following exit codes from the
 * Java application:
 *
 *    0
 *       - Exit normally.
 *    RESTART_LAST_EC = 23
 *       - restart the java VM again with the same arguments as the previous one.
 *    RESTART_NEW_EC  = 24
 *       - restart the java VM again with the arguments taken from the exit data.
 *       The exit data format is a list of arguments separated by '\n'. The Java
 *       application should build this list using the arguments passed to it on
 *       startup. See below.
 *
 * Additionally, if the Java application exits with an exit code other than the
 * ones above, the main launcher will display an error message with the contents
 * of the exit data. If the exit data is empty, a generic error message is
 * displayed. The generic error message shows the exit code and the arguments
 * passed to the Java application.
 *
 * The options that can be specified by the user to the launcher are:
 *  -vm <javaVM>               the Java VM to be used
 *  -os <opSys>                the operating system being run on
 *  -arch <osArch>             the hardware architecture of the OS: x86, sparc, hp9000
 *  -ws <gui>                  the window system to be used: win32, motif, gtk, ...
 *  -nosplash                  do not display the splash screen
 *  <userArgs>                 arguments that are passed along to the Java application
 *                             (i.e, -data <path>, -debug, -console, -consoleLog, etc) 
 *  -vmargs <userVMargs> ...   a list of arguments for the VM itself
 *
 * The -vmargs option and all user specified VM arguments must appear
 * at the end of the command line, after all arguments that are
 * being passed to Java application. 
 *
 * The argument order for the new Java VM process is as follows:
 *
 * <javaVM> <all VM args>
 *     -os <user or default OS value>
 *     -ws <user or default WS value>
 *     -arch <user or default ARCH value>
 *     [-showsplash <splashCommand>]
 *     [-exitdata <exitDataCommand>]
 *     <userArgs>
 *     -vm <javaVM>
 *     -vmargs <all VM args>
 *
 * where:
 *   <all VM args> =
 *     [<defaultVMargs | <userVMargs>]
 *     -cp
 *     <startup jar full path>
 *     org.eclipse.core.launcher.Main
 * 
 * If the -nosplash option is given to the launcher, the java
 * application will not receive the "-showsplash" option or its
 * associated command.
 *
 * See "Main.java" for a simple implementation of the Java
 * application.
 */

#include "eclipseOS.h"
#include "eclipseShm.h"

#ifdef _WIN32
#include <direct.h>
#else
#include <unistd.h>
#include <strings.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#define MAX_PATH_LENGTH   2000
#define MAX_SHARED_LENGTH   (16 * 1024)

/* Global Data */
static _TCHAR*  program     = NULL;       /* full pathname of the program */
static _TCHAR*  homeDir     = NULL;       /* directory where program resides */
static _TCHAR*  javaVM      = NULL;       /* full pathname of the Java VM to run */
static _TCHAR*  sharedID    = NULL;       /* ID for the shared memory */

/* Define the special exit codes returned from Eclipse. */
#define RESTART_LAST_EC    23
#define RESTART_NEW_EC     24

/* Define the maximum time (in seconds) for the splash window to remain visible. */
static _TCHAR*  splashTimeout = _T("600");   /* 10 minutes */

/* Define the required VM arguments (all platforms). */
#define        startupJarName _T("startup.jar")
static _TCHAR*  reqVMarg[] = { _T("-cp"), startupJarName, _T("org.eclipse.core.launcher.Main"), NULL };

/* Define error messages. (non-NLS) */
static _TCHAR* exitMsg = _T("JVM terminated. Exit code=%d\n%s");
static _TCHAR* goVMMsg = _T("Start VM: %s\n");
static _TCHAR* pathMsg = _T("%s\n'%s' in your current PATH");
static _TCHAR* showMsg = _T("Could not load splash bitmap:\n%s");
static _TCHAR* shareMsg = _T("No shared data available.");
static _TCHAR* noVMMsg =
_T("A Java Runtime Environment (JRE) or Java Development Kit (JDK)\n\
must be available in order to run %s. No Java virtual machine\n\
was found after searching the following locations:\n\
%s");

static _TCHAR* homeMsg =
_T("The %s executable launcher was unable to locate its \n\
companion startup.jar file (in the same directory as the executable).");

/* Define constants for the options recognized by the launcher. */
#define CONSOLE      _T("-console")
#define CONSOLELOG   _T("-consoleLog")
#define DEBUG        _T("-debug")
#define OS           _T("-os")
#define OSARCH       _T("-arch")
#define NOSPLASH     _T("-nosplash")
#define SHOWSPLASH   _T("-showsplash")
#define EXITDATA     _T("-exitdata")
#define VM           _T("-vm")
#define WS           _T("-ws")
#define VMARGS       _T("-vmargs")					/* special option processing required */

/* Define the variables to receive the option values. */
static int     needConsole   = 0;				/* True: user wants a console	*/
static int     debug         = 0;				/* True: output debugging info	*/
static int     noSplash      = 0;				/* True: do not show splash win	*/
static _TCHAR*  osArg         = _T(DEFAULT_OS);
static _TCHAR*  osArchArg     = _T(DEFAULT_OS_ARCH);
static _TCHAR*  showSplashArg = NULL;			/* showsplash data (main launcher window) */
static _TCHAR*  exitDataArg   = NULL;
static _TCHAR*  vmName        = NULL;     		/* Java VM that the user wants to run */
static _TCHAR*  wsArg         = _T(DEFAULT_WS);	/* the SWT supported GUI to be used */
static _TCHAR** userVMarg     = NULL;     		/* user specific args for the Java VM  */

/* Define a table for processing command line options. */
typedef struct
{
	_TCHAR*  name;		/* the option recognized by the launcher */
	_TCHAR** value;		/* the variable where the option value is saved */
	int*   flag;		/* the variable that is set if the option is defined */
	int    remove;		/* the number of argments to remove from the list */
} Option;

static Option options[] = {
    { CONSOLE,		NULL,			&needConsole,	0 },
    { CONSOLELOG,	NULL,			&needConsole,	0 },
    { DEBUG,		NULL,			&debug,			0 },
    { NOSPLASH,     NULL,           &noSplash,		1 },
    { OS,			&osArg,			NULL,			2 },
    { OSARCH,		&osArchArg,		NULL,			2 },
    { SHOWSPLASH,   &showSplashArg,	NULL,			2 },
    { EXITDATA,		&exitDataArg,	NULL,			2 },
    { VM,           &vmName,		NULL,			2 },
    { WS,			&wsArg,			NULL,			2 } };
static int optionsSize = (sizeof(options) / sizeof(options[0]));

/* Local methods */
static void   parseArgs( int* argc, _TCHAR* argv[] );
static _TCHAR** parseArgList( _TCHAR *data );
static void   freeArgList( _TCHAR** data );
static _TCHAR** getVMCommand( int argc, _TCHAR* argv[] );
       _TCHAR*  findCommand( _TCHAR* command );
static _TCHAR*  formatVmCommandMsg( _TCHAR* args[] );
static _TCHAR*  getInstallDir();

#ifdef _WIN32
#ifdef UNICODE
extern int main(int, char**);
int mainW(int, wchar_t**);
int wmain( int argc, wchar_t** argv ) {
	OSVERSIONINFOW info;
	info.dwOSVersionInfoSize = sizeof(OSVERSIONINFOW);
	/*
	* If the OS supports UNICODE functions, run the UNICODE version
	* of the main function. Otherwise, convert the arguments to
	* MBCS and run the ANSI version of the main function.
	*/
	if (!GetVersionExW (&info)) {
		int i, result;
		char **newArgv = malloc(argc * sizeof(char *));
		for (i=0; i<argc; i++) {
			wchar_t *oldArg = argv[i];
			int byteCount = WideCharToMultiByte (CP_ACP, 0, oldArg, -1, NULL, 0, NULL, NULL);
			char *newArg  = malloc(byteCount+1);
			newArg[byteCount] = 0;
			WideCharToMultiByte (CP_ACP, 0, oldArg, -1, newArg, byteCount, NULL, NULL);
			newArgv[i] = newArg;
		}
		result = main(argc, newArgv);
		for (i=0; i<argc; i++) {
			free(newArgv[i]);
		}
		free(newArgv);
		return result;
	}
	return mainW(argc, argv);
}
#define main mainW
#endif /* UNICODE */
#endif /* _WIN32 */

int main( int argc, _TCHAR* argv[] )
{
	_TCHAR*   splashBitmap;
    _TCHAR*   ch;
    _TCHAR*   data;
    _TCHAR*   shippedVM    = NULL;
    _TCHAR*   vmSearchPath = NULL;
    _TCHAR**  vmCommand = NULL;
    _TCHAR**  vmCommandList = NULL;
    _TCHAR**  vmCommandArgs = NULL;
    _TCHAR*   vmCommandMsg = NULL;
    _TCHAR*   errorMsg;
    int       exitCode;
    
	/* 
	 * Strip off any extroneous <CR> from the last argument. If a shell script
	 * on Linux is created in DOS format (lines end with <CR><LF>), the C-shell
	 * does not strip off the <CR> and hence the argument is bogus and may 
	 * not be recognized by the launcher or eclipse itself.
	 */
	 ch = _tcschr( argv[ argc - 1 ], _T('\r') );
	 if (ch != NULL)
	 {
	     *ch = _T('\0');
	 }
	 
	/* Determine the full pathname of this program. */
    program = findCommand( argv[0] );
    if (program == NULL)
    {
#ifdef _WIN32
    	program = malloc( MAX_PATH_LENGTH + 1 );
    	GetModuleFileName( NULL, program, MAX_PATH_LENGTH );
#else
    	program = malloc( strlen( argv[0] ) + 1 );
    	strcpy( program, argv[0] );
#endif
    }

    /* Parse command line arguments (looking for the VM to use). */
    parseArgs( &argc, argv );

    /* Initialize the window system. */
    initWindowSystem( &argc, argv, (showSplashArg != NULL) );

    /* Find the home directory where the Eclipse is installed. */
    homeDir = getInstallDir();
    if (homeDir == NULL)
    {
        errorMsg = malloc( (_tcslen(homeMsg) + _tcslen(getOfficialName()) + 10) * sizeof(_TCHAR) );
        _stprintf( errorMsg, homeMsg, getOfficialName() );
        displayMessage( errorMsg );
        free( errorMsg );
    	exit( 1 );
    }

    /* If the exit data option was given, set exit data */
    if (exitDataArg != NULL)
    {
       	/* If an extra argument was given, use it as the exit data, otherwise clear exit data */
       	data = argc > 1 ? argv[1] : NULL;
       	if (data != NULL && _tcslen( data ) > MAX_SHARED_LENGTH - 1)
       	{
       		exitCode = EINVAL;       		
       	}
       	else {
	        exitCode = setSharedData( exitDataArg, data );
    	}
   	    if (exitCode != 0 && debug) displayMessage( shareMsg );
		exit( exitCode );
    }

    /* If the showsplash option was given */
    if (showSplashArg != NULL)
    {
    	/* If an extra argument was given, pass it as the image to display. */
    	splashBitmap = (argc > 1 ? argv[1] : NULL);
    	exitCode = showSplash( showSplashArg, homeDir, splashBitmap );
    	if (exitCode && debug)
    	{
    		if (splashBitmap == NULL)
    			splashBitmap = homeDir;
        	errorMsg = malloc( (_tcslen(showMsg) + _tcslen(splashBitmap) + 10) * sizeof(_TCHAR) );
        	_stprintf( errorMsg, showMsg, splashBitmap );
        	displayMessage( errorMsg );
        	free( errorMsg );
     	}
    	exit( exitCode );
    }

    /* If the user did not specify a VM to be used */
    if (vmName == NULL)
    {
    	/* Determine which type of VM should be used. */
    	vmName = ((debug || needConsole) ? consoleVM : defaultVM);

        /* Try to find the VM shipped with eclipse. */
        shippedVM = malloc( (_tcslen( homeDir ) + _tcslen( shippedVMDir ) + _tcslen( vmName ) + 10) * sizeof(_TCHAR) );
        _stprintf( shippedVM, _T("%s%s%s"), homeDir, shippedVMDir, vmName );
        javaVM = findCommand( shippedVM );

        /* Format a message to indicate the default VM search path. */
        vmSearchPath = malloc( (_tcslen( pathMsg ) + _tcslen( shippedVM ) + _tcslen( vmName ) + 10) * sizeof(_TCHAR) );
        _stprintf( vmSearchPath, pathMsg, shippedVM, vmName );
        free( shippedVM );
        shippedVM = NULL;
	}

	/* If a Java VM has not been found yet */
	if (javaVM == NULL)
	{
		/* Either verify the VM specified by the user or
		   attempt to find the VM in the user's PATH. */
		javaVM = findCommand( vmName );

		/* If the VM was not found, display a message and exit. */
		if (javaVM == NULL)
		{
			if (vmSearchPath != NULL) vmName = vmSearchPath; /* used default VM searching */
        	errorMsg = malloc( (_tcslen(noVMMsg) + _tcslen(getOfficialName()) + _tcslen(vmName) + 10) * sizeof(_TCHAR) );
        	_stprintf( errorMsg, noVMMsg, getOfficialName(), vmName );
        	displayMessage( errorMsg );
        	free( errorMsg );
        	exit(1);
		}
	}

    if (createSharedData( &sharedID, MAX_SHARED_LENGTH )) {
        if (debug) {
   			if (debug) displayMessage( shareMsg );
        }
    }

    /* Get the command to start the Java VM. */
    vmCommandArgs = getVMCommand( argc, argv );

    /* While the Java VM should be restarted */
    vmCommand = vmCommandArgs;
    while (vmCommand != NULL)
    {
    	vmCommandMsg = formatVmCommandMsg( vmCommand );
    	if (debug) _tprintf( goVMMsg, vmCommandMsg );
    	exitCode = startJavaVM( vmCommand );
        switch( exitCode ) {
            case 0:
                vmCommand = NULL;
            	break;
            case RESTART_LAST_EC:
            	break;
            case RESTART_NEW_EC:
                if (getSharedData( sharedID, &data ) == 0) {
			    	if (vmCommandList != NULL) freeArgList( vmCommandList );
                    vmCommand = vmCommandList = parseArgList( data );
                } else {
                	vmCommand = NULL;
                    if (debug) displayMessage( shareMsg );
                }
                break;
			default:
                vmCommand = NULL;
                errorMsg = NULL;
                if (getSharedData( sharedID, &errorMsg ) == 0) {
                    if (_tcslen( errorMsg ) == 0) {
                	    free( errorMsg );
                	    errorMsg = NULL;
                    }
                } else {
                    if (debug) displayMessage( shareMsg );
                }
                if (errorMsg == NULL) {
	                errorMsg = malloc( (_tcslen(exitMsg) + _tcslen(vmCommandMsg) + 10) * sizeof(_TCHAR) );
	                _stprintf( errorMsg, exitMsg, exitCode, vmCommandMsg );
                }
	            displayMessage( errorMsg );
	            free( errorMsg );
                break;
        }
        free( vmCommandMsg );
    }

    /* Cleanup time. */
    free( homeDir );
    free( program );
    if ( vmSearchPath != NULL ) free( vmSearchPath );
    if ( vmCommandList != NULL ) freeArgList( vmCommandList );
    if ( sharedID != NULL ) {
        if (destroySharedData( sharedID ) != 0) {
           if (debug) displayMessage( shareMsg );
        }
        free( sharedID );
    }

    return 0;
}


/*
 * Parse arguments of the command.
 */
static void parseArgs( int* pArgc, _TCHAR* argv[] )
{
	Option* option;
    int     remArgs;
    int     index;
    int     i;

    /* Ensure the list of user argument is NULL terminated. */
    argv[ *pArgc ] = NULL;

	/* For each user defined argument (excluding the program) */
    for (index = 1; index < *pArgc; index++){
        remArgs = 0;

        /* Find the corresponding argument is a option supported by the launcher */
        option = NULL;
        for (i = 0; option == NULL && i < optionsSize; i++)
        {
        	if (_tcsicmp( argv[ index ], options[ i ].name ) == 0)
        	    option = &options[ i ];
       	}

       	/* If the option is recognized by the launcher */
       	if (option != NULL)
       	{
       		/* If the option requires a value and there is one, extract the value. */
       		if (option->value != NULL && (index+1) < *pArgc)
       			*option->value = argv[ index+1 ];

       		/* If the option requires a flag to be set, set it. */
       		if (option->flag != NULL)
       			*option->flag = 1;
       		remArgs = option->remove;
       	}

        /* All of the remaining arguments are user VM args. */
        else if (_tcsicmp( argv[ index ], VMARGS ) == 0)
        {
            userVMarg = &argv[ index+1 ];
            argv[ index ] = NULL;
            *pArgc = index;
        }

		/* Remove any matched arguments from the list. */
        if (remArgs > 0)
        {
            for (i = (index + remArgs); i <= *pArgc; i++)
            {
                argv[ i - remArgs ] = argv[ i ];
            }
            index--;
            *pArgc -= remArgs;
        }
    }
}

/*
 * Free the memory allocated by parseArgList().
 */
static void freeArgList( _TCHAR** data ) {
	if (data == NULL) return;
	free( data [0] );
	free( data );
}

/*
 * Parse the data into a list of arguments separarted by \n.
 *
 * The list of strings returned by this function must be freed with
 * freeArgList().
 */
static _TCHAR** parseArgList( _TCHAR* data ) {
    int totalArgs = 0, dst = 0, length;
    _TCHAR *ch1, *ch2, **execArg;
    length = _tcslen( data );
    ch1 = ch2 = data;
    while ((ch2 = _tcschr( ch1, _T('\n') )) != NULL) {
    	totalArgs++;
    	ch1 = ch2 + 1;
    }
    if (ch1 != data + length) totalArgs++;
    execArg = malloc( (totalArgs + 1) * sizeof( _TCHAR* ) );
    ch1 = ch2 = data;
    while ((ch2 = _tcschr( ch1, _T('\n') )) != NULL) {
    	execArg[ dst++ ] = ch1;
    	ch2[ 0 ] = _T('\0');
    	ch1 = ch2 + 1;
    }
    if (ch1 != data + length) execArg[ dst++ ] = ch1;
    execArg[ dst++ ] = NULL;
    return execArg;
}

/*
 * Find the absolute pathname to where a command resides.
 *
 * The string returned by the function must be freed.
 */
#define EXTRA 20
_TCHAR* findCommand( _TCHAR* command )
{
    _TCHAR*  cmdPath;
    int    length;
    _TCHAR*  ch;
    _TCHAR*  dir;
    _TCHAR*  path;
    struct _stat stats;

    /* If the command was an abolute pathname, use it as is. */
    if (command[0] == dirSeparator ||
       (_tcslen( command ) > 2 && command[1] == _T(':')))
    {
        length = _tcslen( command );
        cmdPath = malloc( (length + EXTRA) * sizeof(_TCHAR) ); /* add extra space for a possible ".exe" extension */
        _tcscpy( cmdPath, command );
    }

    else
    {
        /* If the command string contains a path separator */
        if (_tcschr( command, dirSeparator ) != NULL)
        {
            /* It must be relative to the current directory. */
            length = MAX_PATH_LENGTH + EXTRA + _tcslen( command );
            cmdPath = malloc( length * sizeof (_TCHAR));
            _tgetcwd( cmdPath, length );
            if (cmdPath[ _tcslen( cmdPath ) - 1 ] != dirSeparator)
            {
                length = _tcslen( cmdPath );
                cmdPath[ length ] = dirSeparator;
                cmdPath[ length+1 ] = _T('\0');
            }
            _tcscat( cmdPath, command );
        }

        /* else the command must be in the PATH somewhere */
        else
        {
            /* Get the directory PATH where executables reside. */
            path = _tgetenv( _T("PATH") );
            length = _tcslen( path ) + _tcslen( command ) + MAX_PATH_LENGTH;
            cmdPath = malloc( length * sizeof(_TCHAR));

            /* Foreach directory in the PATH */
            dir = path;
            while (dir != NULL && *dir != _T('\0'))
            {
                ch = _tcschr( dir, pathSeparator );
                if (ch == NULL)
                {
                    _tcscpy( cmdPath, dir );
                }
                else
                {
                    length = ch - dir;
                    _tcsncpy( cmdPath, dir, length );
                    cmdPath[ length ] = _T('\0');
                    ch++;
                }
                dir = ch; /* advance for the next iteration */

                /* Determine if the executable resides in this directory. */
                if (cmdPath[0] == _T('.') &&
                   (_tcslen(cmdPath) == 1 || (_tcslen(cmdPath) == 2 && cmdPath[1] == dirSeparator)))
                {
                	_tgetcwd( cmdPath, MAX_PATH_LENGTH );
                }
                if (cmdPath[ _tcslen( cmdPath ) - 1 ] != dirSeparator)
                {
                    length = _tcslen( cmdPath );
                    cmdPath[ length ] = dirSeparator;
                    cmdPath[ length+1 ] = _T('\0');
                }
                _tcscat( cmdPath, command );

                /* If the file is not a directory and can be executed */
                if (_tstat( cmdPath, &stats ) == 0 && (stats.st_mode & S_IFREG) != 0)
                {
                    /* Stop searching */
                    dir = NULL;
                }
            }
        }
    }

#ifdef _WIN32
	/* If the command does not exist */
    if (_tstat( cmdPath, &stats ) != 0 || (stats.st_mode & S_IFREG) == 0)
    {
    	/* If the command does not end with .exe, append it an try again. */
    	length = _tcslen( cmdPath );
    	if (length > 4 && _tcsicmp( &cmdPath[ length - 4 ], _T(".exe") ) != 0)
    	    _tcscat( cmdPath, _T(".exe") );
    }
#endif

    /* Verify the resulting command actually exists. */
    if (_tstat( cmdPath, &stats ) != 0 || (stats.st_mode & S_IFREG) == 0)
    {
        free( cmdPath );
        cmdPath = NULL;
    }

    /* Return the absolute command pathname. */
    return cmdPath;
}

/*
 * Get the command and arguments to start the Java VM.
 *
 * Memory allocated by this function is assumed to be
 * deallocated when the program terminates.
 *
 * Some of the arguments returned by this function were
 * passed directly from the main( argv ) array so they
 * should not be deallocated.
 */
static _TCHAR** getVMCommand( int argc, _TCHAR* argv[] )
{
	_TCHAR** defVMarg;
    int     nDefVMarg = 0;
    int     nReqVMarg = 0;
    int     nUserVMarg = 0;
    int     totalArgs;
    _TCHAR** execArg;
    _TCHAR*  jarFile = NULL;
    _TCHAR*  splashExec;
	_TCHAR*  exitDataExec;
    int     src;
    int     dst;

 	/* Calculate the number of user VM arguments. */
 	if (userVMarg != NULL)
 	{
	 	while (userVMarg[ nUserVMarg ] != NULL)
 			nUserVMarg++;
 	}

 	/* Calculate the number of default VM arguments. */
 	defVMarg = getArgVM( javaVM );
 	while (defVMarg[ nDefVMarg ] != NULL)
 		nDefVMarg++;

 	/* Calculate the number of required VM arguments. */
 	while (reqVMarg[ nReqVMarg ] != NULL)
 		nReqVMarg++;

    /* Allocate the arg list for the exec call.
     *  (VM + userVMargs + defaultVMargs + requiredVMargs + OS <os> + WS <ws> + ARCH <arch>
     *      + SHOWSPLASH <cmd> + EXITDATA <cmd> + argv[] + VM + <vm> + VMARGS + userVMargs + defaultVMargs + requiredVMargs
     *      + NULL)
     */
    totalArgs  = 1 + nUserVMarg + nDefVMarg + nReqVMarg + 2 + 2 + 2 + 2 + 2 + argc + 2 + 1 + nUserVMarg + nDefVMarg + nReqVMarg + 1;
	execArg = malloc( totalArgs * sizeof( _TCHAR* ) );
    dst = 0;
    execArg[ dst++ ] = javaVM;

    /* If the user specified "-vmargs", add them instead of the default VM args. */
    if (userVMarg != NULL)
    {
    	for (src = 0; src < nUserVMarg; src++)
	    	execArg[ dst++ ] = userVMarg[ src ];
	}
	else
	{
    	for (src = 0; src < nDefVMarg; src++)
	    	execArg[ dst++ ] = defVMarg[ src ];
	}

    /* For each required VM arg */
	jarFile = malloc( (_tcslen( homeDir ) + _tcslen( startupJarName ) + 1) * sizeof( _TCHAR ) );
    jarFile = _tcscpy( jarFile, homeDir );
  	jarFile = _tcscat( jarFile, startupJarName );
	for (src = 0; src < nReqVMarg; src++)
    {
    	/* If the argument is not the startup jar, use it as is. */
     	if (_tcscmp( reqVMarg[ src ], startupJarName ) != 0)
     	{
	        execArg[ dst++ ] = reqVMarg[ src ];
	    }

	    /* else use the absolute path of the jar file. */
	    else
	    {
	        execArg[ dst++ ] = jarFile;
	    }
    }

	/* Append the required options. */
    execArg[ dst++ ] = OS;
    execArg[ dst++ ] = osArg;
    execArg[ dst++ ] = WS;
    execArg[ dst++ ] = wsArg;
    execArg[ dst++ ] = OSARCH;
    execArg[ dst++ ] = osArchArg;

	/* Append the show splash window command, if defined. */
    if (!noSplash)
    {
        execArg[ dst++ ] = SHOWSPLASH;
		splashExec = malloc( (_tcslen( program ) + _tcslen(SHOWSPLASH) + _tcslen( splashTimeout ) + 3) * sizeof(_TCHAR) );
		_stprintf( splashExec, _T("%s %s %s"), program, SHOWSPLASH, splashTimeout );
	    execArg[ dst++ ] = splashExec;
    }

	/* Append the exit data command. */
	if (sharedID) {
	    execArg[ dst++ ] = EXITDATA;
	    exitDataExec = malloc( (_tcslen( program ) + _tcslen( EXITDATA ) + _tcslen( sharedID ) + 3) * sizeof(_TCHAR) );
	    _stprintf( exitDataExec, _T("%s %s %s"), program, EXITDATA, sharedID );
	    execArg[ dst++ ] = exitDataExec;
	}

	/* Append the remaining user defined arguments. */
    for (src = 1; src < argc; src++)
    {
        execArg[ dst++ ] = argv[ src ];
    }

    /* Append VM and VMARGS to be able to relaunch using exit data. */
	execArg[ dst++ ] = VM;
	execArg[ dst++ ] = javaVM;
    execArg[ dst++ ] = VMARGS;
    /* If the user specified "-vmargs", add them instead of the default VM args. */
    if (userVMarg != NULL)
    {
    	for (src = 0; src < nUserVMarg; src++)
	    	execArg[ dst++ ] = userVMarg[ src ];
	}
	else
	{
    	for (src = 0; src < nDefVMarg; src++)
	    	execArg[ dst++ ] = defVMarg[ src ];
	}
    /* For each required VM arg */
    for (src = 0; src < nReqVMarg; src++)
    {
    	/* If the argument is not the startup jar, use it as is. */
     	if (_tcscmp( reqVMarg[ src ], startupJarName ) != 0)
     	{
	        execArg[ dst++ ] = reqVMarg[ src ];
	    }

	    /* else use the absolute path of the jar file. */
	    else
	    {
	        execArg[ dst++ ] = jarFile;
	    }
    }

    execArg[ dst++ ] = NULL;

	return execArg;
 }

 /* Format the JVM start command for error messages
  *
  * This method formats a string with the JVM start command (and all arguments)
  * that can be used in displaying error messages. The string returned from this
  * method is probably not NLS compliant and must be deallocated by the caller.
  */
static _TCHAR* formatVmCommandMsg( _TCHAR* args[] )
{
	int   index;
    int   length;
    _TCHAR* ch;
    _TCHAR* message;

	/* Determine the length of the message buffer. */
	length = 0;
	for (index = 0; args[index] != NULL; index++)
	{
		length += _tcslen(args[index]) + 1;
	}
	message = malloc( (length + 5) * sizeof(_TCHAR) );
	
	/* Format the message such that options (args starting with '-') begin
	   on a new line. Otherwise, the Motif MessageBox does not automatically wrap
	   the messages and the message window can extend beyond both sides of the display. */
	ch = message;
	for (index = 0; args[index] != NULL; index++)
	{
		if (args[index][0] == _T('-') && *(ch-1) == _T(' '))
			*(ch-1) = _T('\n');
		_tcscpy( ch, args[index] );
		ch += _tcslen( args[index] );
		*ch++ = _T(' ');
	}
	*ch = _T('\0');

	return message;
}

/* Determine the Installation Directory
 *
 * This function takes the directory where program executable resides and
 * determines the installation directory. The installation directory must
 * contain the startup JAR file and the executable must be either in the
 * install directory or in a subdirectory.
 */
static _TCHAR* getInstallDir( )
{
	_TCHAR*  ch;
	_TCHAR*  installDir;
	_TCHAR   path[ MAX_PATH_LENGTH ];
    struct _stat stats;

	/* Start with the directory where the executable resides. */
    installDir = malloc( (_tcslen( program ) + 1) * sizeof(_TCHAR) );
    _tcscpy( installDir, program );
    ch = _tcsrchr( installDir, dirSeparator );

    /* For each directory in the hierarchy */
    while (ch != NULL)
    {
    	/* Format the pathname of the startup jar file. */
    	*(ch+1) = _T('\0');
    	_stprintf( path, _T("%s%s"), installDir, startupJarName );

    	/* If the jar file exists */
    	if (_tstat( path, &stats ) == 0 && (stats.st_mode & S_IFREG) != 0)
    	{
    		return installDir;
    	}
    	else
    	{
    		/* Try the next directory up in the hierarchy. */
    		*ch = _T('\0');
    		ch = _tcsrchr( installDir, dirSeparator );
    	}
    }

    free( installDir );
    return NULL;
}

