/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fileformat;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PushbackReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.jabref.logic.bibtex.FieldContentParser;
import org.jabref.logic.importer.ImportFormatPreferences;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.Parser;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.importer.util.MetaDataParser;
import org.jabref.logic.l10n.Localization;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.KeyCollisionException;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibtexString;
import org.jabref.model.entry.CustomEntryType;
import org.jabref.model.entry.EntryType;
import org.jabref.model.entry.FieldProperty;
import org.jabref.model.entry.InternalBibtexFields;
import org.jabref.model.metadata.MetaData;
import org.jabref.model.util.FileUpdateMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BibtexParser
implements Parser {
    private static final Logger LOGGER = LoggerFactory.getLogger(BibtexParser.class);
    private static final Integer LOOKAHEAD = 64;
    private final FieldContentParser fieldContentParser;
    private final Deque<Character> pureTextFromFile = new LinkedList<Character>();
    private final ImportFormatPreferences importFormatPreferences;
    private PushbackReader pushbackReader;
    private BibDatabase database;
    private Map<String, EntryType> entryTypes;
    private boolean eof;
    private int line = 1;
    private ParserResult parserResult;
    private MetaDataParser metaDataParser;

    public BibtexParser(ImportFormatPreferences importFormatPreferences, FileUpdateMonitor fileMonitor) {
        this.importFormatPreferences = Objects.requireNonNull(importFormatPreferences);
        this.fieldContentParser = new FieldContentParser(importFormatPreferences.getFieldContentParserPreferences());
        this.metaDataParser = new MetaDataParser(fileMonitor);
    }

    public static Optional<BibEntry> singleFromString(String bibtexString, ImportFormatPreferences importFormatPreferences, FileUpdateMonitor fileMonitor) throws ParseException {
        List<BibEntry> entries = new BibtexParser(importFormatPreferences, fileMonitor).parseEntries(bibtexString);
        if (entries == null || entries.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of((BibEntry)entries.iterator().next());
    }

    @Override
    public List<BibEntry> parseEntries(InputStream inputStream) throws ParseException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
        return this.parseEntries(reader);
    }

    public List<BibEntry> parseEntries(Reader reader) throws ParseException {
        try {
            return this.parse(reader).getDatabase().getEntries();
        }
        catch (IOException e) {
            throw new ParseException(e);
        }
    }

    public List<BibEntry> parseEntries(String bibtexString) throws ParseException {
        return this.parseEntries(new StringReader(bibtexString));
    }

    public Optional<BibEntry> parseSingleEntry(String bibtexString) throws ParseException {
        return this.parseEntries(bibtexString).stream().findFirst();
    }

    public ParserResult parse(Reader in) throws IOException {
        Objects.requireNonNull(in);
        this.pushbackReader = new PushbackReader(in, LOOKAHEAD);
        this.initializeParserResult();
        this.parseDatabaseID();
        this.skipWhitespace();
        try {
            return this.parseFileContent();
        }
        catch (KeyCollisionException kce) {
            throw new IOException("Duplicate ID in bibtex file: " + kce);
        }
    }

    private void initializeParserResult() {
        this.database = new BibDatabase();
        this.entryTypes = new HashMap<String, EntryType>();
        this.parserResult = new ParserResult(this.database, new MetaData(), this.entryTypes);
    }

    private void parseDatabaseID() throws IOException {
        while (!this.eof) {
            this.skipWhitespace();
            char c = (char)this.read();
            if (c == '%') {
                this.skipWhitespace();
                String label = this.parseTextToken().trim();
                if (!label.equals("DBID:")) continue;
                this.skipWhitespace();
                this.database.setSharedDatabaseID(this.parseTextToken().trim());
                continue;
            }
            if (c != '@') continue;
            this.unread(c);
            break;
        }
    }

    private ParserResult parseFileContent() throws IOException {
        boolean found;
        HashMap<String, String> meta = new HashMap<String, String>();
        while (!this.eof && (found = this.consumeUncritically('@'))) {
            this.skipWhitespace();
            String entryType = this.parseTextToken().toLowerCase(Locale.ROOT).trim();
            if ("preamble".equals(entryType)) {
                this.database.setPreamble(this.parsePreamble());
                this.skipOneNewline();
                this.dumpTextReadSoFarToString();
            } else if ("string".equals(entryType)) {
                this.parseBibtexString();
            } else if ("comment".equals(entryType)) {
                this.parseJabRefComment(meta);
            } else {
                this.parseAndAddEntry(entryType);
            }
            this.skipWhitespace();
        }
        try {
            this.parserResult.setMetaData(this.metaDataParser.parse(meta, this.importFormatPreferences.getKeywordSeparator()));
        }
        catch (ParseException exception) {
            this.parserResult.addException(exception);
        }
        this.parseRemainingContent();
        return this.parserResult;
    }

    private void parseRemainingContent() {
        this.database.setEpilog(this.dumpTextReadSoFarToString().trim());
    }

    private void parseAndAddEntry(String type) {
        try {
            String commentsAndEntryTypeDefinition = this.dumpTextReadSoFarToString();
            BibEntry entry = this.parseEntry(type);
            entry.setCommentsBeforeEntry(commentsAndEntryTypeDefinition.substring(0, commentsAndEntryTypeDefinition.lastIndexOf(64)));
            entry.setParsedSerialization(commentsAndEntryTypeDefinition + this.dumpTextReadSoFarToString());
            boolean duplicateKey = this.database.insertEntry(entry);
            if (duplicateKey) {
                this.parserResult.addDuplicateKey(entry.getCiteKey());
            }
        }
        catch (IOException ex) {
            LOGGER.debug("Could not parse entry", ex);
            this.parserResult.addWarning(Localization.lang("Error occurred when parsing entry", new String[0]) + ": '" + ex.getMessage() + "'. " + Localization.lang("Skipped entry.", new String[0]));
        }
    }

    private void parseJabRefComment(Map<String, String> meta) {
        StringBuilder buffer = null;
        try {
            buffer = this.parseBracketedTextExactly();
        }
        catch (IOException e) {
            LOGGER.info("Found unbracketed comment");
            return;
        }
        String comment = buffer.toString().replaceAll("[\\x0d\\x0a]", "");
        if (comment.substring(0, Math.min(comment.length(), "jabref-meta: ".length())).equals("jabref-meta: ")) {
            String rest;
            int pos;
            if (comment.substring(0, "jabref-meta: ".length()).equals("jabref-meta: ") && (pos = (rest = comment.substring("jabref-meta: ".length())).indexOf(58)) > 0) {
                meta.put(rest.substring(0, pos), rest.substring(pos + 1));
                this.dumpTextReadSoFarToString();
            }
        } else if (comment.substring(0, Math.min(comment.length(), "jabref-entrytype: ".length())).equals("jabref-entrytype: ")) {
            Optional<CustomEntryType> typ = CustomEntryType.parse(comment);
            if (typ.isPresent()) {
                this.entryTypes.put(typ.get().getName(), typ.get());
            } else {
                this.parserResult.addWarning(Localization.lang("Ill-formed entrytype comment in BIB file", new String[0]) + ": " + comment);
            }
            this.dumpTextReadSoFarToString();
        }
    }

    private void parseBibtexString() throws IOException {
        BibtexString bibtexString = this.parseString();
        bibtexString.setParsedSerialization(this.dumpTextReadSoFarToString());
        try {
            this.database.addString(bibtexString);
        }
        catch (KeyCollisionException ex) {
            this.parserResult.addWarning(Localization.lang("Duplicate string name", new String[0]) + ": " + bibtexString.getName());
        }
    }

    private String dumpTextReadSoFarToString() {
        String result = this.getPureTextFromFile();
        int indexOfAt = result.indexOf("@");
        if (indexOfAt == -1) {
            return this.purgeEOFCharacters(result);
        }
        if (result.contains("DBID:")) {
            return this.purge(result, "DBID:");
        }
        if (result.contains("Encoding: ")) {
            return this.purge(result, "Encoding: ");
        }
        return result;
    }

    private String purge(String context, String stringToPurge) {
        int runningIndex;
        int indexOfAt = context.indexOf("@");
        for (runningIndex = context.indexOf(stringToPurge); runningIndex < indexOfAt && context.charAt(runningIndex) != '\n'; ++runningIndex) {
            if (context.charAt(runningIndex) != '\r') continue;
            if (context.charAt(runningIndex + 1) != '\n') break;
            ++runningIndex;
            break;
        }
        return context.substring(runningIndex + 1);
    }

    private String getPureTextFromFile() {
        StringBuilder entry = new StringBuilder();
        while (!this.pureTextFromFile.isEmpty()) {
            entry.append(this.pureTextFromFile.pollFirst());
        }
        return entry.toString();
    }

    private String purgeEOFCharacters(String input) {
        StringBuilder remainingText = new StringBuilder();
        char[] cArray = input.toCharArray();
        int n = cArray.length;
        for (int i = 0; i < n; ++i) {
            Character character = Character.valueOf(cArray[i]);
            if (this.isEOFCharacter(character.charValue())) continue;
            remainingText.append(character);
        }
        return remainingText.toString();
    }

    private void skipWhitespace() throws IOException {
        int character;
        do {
            if (!this.isEOFCharacter(character = this.read())) continue;
            this.eof = true;
            return;
        } while (Character.isWhitespace((char)character));
        this.unread(character);
    }

    private void skipSpace() throws IOException {
        int character;
        do {
            if (!this.isEOFCharacter(character = this.read())) continue;
            this.eof = true;
            return;
        } while ((char)character == ' ');
        this.unread(character);
    }

    private void skipOneNewline() throws IOException {
        this.skipSpace();
        if (this.peek() == 13) {
            this.read();
        }
        if (this.peek() == 10) {
            this.read();
        }
    }

    private boolean isEOFCharacter(int character) {
        return character == -1 || character == 65535;
    }

    private String skipAndRecordWhitespace(int character) throws IOException {
        int nextCharacter;
        StringBuilder stringBuilder = new StringBuilder();
        if (character != 32) {
            stringBuilder.append((char)character);
        }
        while (true) {
            if (this.isEOFCharacter(nextCharacter = this.read())) {
                this.eof = true;
                return stringBuilder.toString();
            }
            if (!Character.isWhitespace((char)nextCharacter)) break;
            if (nextCharacter == 32) continue;
            stringBuilder.append((char)nextCharacter);
        }
        this.unread(nextCharacter);
        return stringBuilder.toString();
    }

    private int peek() throws IOException {
        int character = this.read();
        this.unread(character);
        return character;
    }

    private int read() throws IOException {
        int character = this.pushbackReader.read();
        if (!this.isEOFCharacter(character)) {
            this.pureTextFromFile.offerLast(Character.valueOf((char)character));
        }
        if (character == 10) {
            ++this.line;
        }
        return character;
    }

    private void unread(int character) throws IOException {
        if (character == 10) {
            --this.line;
        }
        this.pushbackReader.unread(character);
        if (this.pureTextFromFile.getLast().charValue() == character) {
            this.pureTextFromFile.pollLast();
        }
    }

    private BibtexString parseString() throws IOException {
        this.skipWhitespace();
        this.consume('{', '(');
        this.skipWhitespace();
        LOGGER.debug("Parsing string name");
        String name = this.parseTextToken();
        LOGGER.debug("Parsed string name");
        this.skipWhitespace();
        LOGGER.debug("Now the contents");
        this.consume('=');
        String content = this.parseFieldContent(name);
        LOGGER.debug("Now I'm going to consume a }");
        this.consume('}', ')');
        this.skipOneNewline();
        LOGGER.debug("Finished string parsing.");
        return new BibtexString(name, content);
    }

    private String parsePreamble() throws IOException {
        this.skipWhitespace();
        return this.parseBracketedText();
    }

    private BibEntry parseEntry(String entryType) throws IOException {
        BibEntry result = new BibEntry(entryType);
        this.skipWhitespace();
        this.consume('{', '(');
        int character = this.peek();
        if (character != 10 && character != 13) {
            this.skipWhitespace();
        }
        String key = this.parseKey();
        result.setCiteKey(key);
        this.skipWhitespace();
        while ((character = this.peek()) != 125 && character != 41) {
            if (character == 44) {
                this.consume(',');
            }
            this.skipWhitespace();
            character = this.peek();
            if (character == 125 || character == 41) break;
            this.parseField(result);
        }
        this.consume('}', ')');
        this.skipOneNewline();
        return result;
    }

    private void parseField(BibEntry entry) throws IOException {
        String key = this.parseTextToken().toLowerCase(Locale.ROOT);
        this.skipWhitespace();
        this.consume('=');
        String content = this.parseFieldContent(key);
        if (!content.isEmpty()) {
            if (entry.hasField(key)) {
                if (InternalBibtexFields.getFieldProperties(key).contains((Object)FieldProperty.PERSON_NAMES)) {
                    entry.setField(key, entry.getField(key).get() + " and " + content);
                } else if ("keywords".equals(key)) {
                    entry.addKeyword(content, this.importFormatPreferences.getKeywordSeparator());
                }
            } else {
                entry.setField(key, content);
            }
        }
    }

    private String parseFieldContent(String key) throws IOException {
        int character;
        this.skipWhitespace();
        StringBuilder value = new StringBuilder();
        while ((character = this.peek()) != 44 && character != 125 && character != 41) {
            StringBuilder text;
            if (this.eof) {
                throw new IOException("Error in line " + this.line + ": EOF in mid-string");
            }
            if (character == 34) {
                text = this.parseQuotedFieldExactly();
                value.append(this.fieldContentParser.format(text, key));
            } else if (character == 123) {
                text = this.parseBracketedTextExactly();
                value.append(this.fieldContentParser.format(text, key));
            } else if (Character.isDigit((char)character)) {
                String number = this.parseTextToken();
                value.append(number);
            } else if (character == 35) {
                this.consume('#');
            } else {
                String textToken = this.parseTextToken();
                if (textToken.isEmpty()) {
                    throw new IOException("Error in line " + this.line + " or above: Empty text token.\nThis could be caused by a missing comma between two fields.");
                }
                value.append('#').append(textToken).append('#');
            }
            this.skipWhitespace();
        }
        return value.toString();
    }

    private String parseTextToken() throws IOException {
        int character;
        StringBuilder token = new StringBuilder(20);
        while (true) {
            if ((character = this.read()) == -1) {
                this.eof = true;
                return token.toString();
            }
            if (!Character.isLetterOrDigit((char)character) && ":-_*+./'".indexOf(character) < 0) break;
            token.append((char)character);
        }
        this.unread(character);
        return token.toString();
    }

    private String fixKey() throws IOException {
        char currentChar;
        StringBuilder key = new StringBuilder();
        int lookaheadUsed = 0;
        do {
            currentChar = (char)this.read();
            key.append(currentChar);
        } while (currentChar != ',' && currentChar != '\n' && currentChar != '=' && ++lookaheadUsed < LOOKAHEAD);
        this.unread(currentChar);
        key.deleteCharAt(key.length() - 1);
        switch (currentChar) {
            case '=': {
                key = key.reverse();
                boolean matchedAlpha = false;
                for (int i = 0; i < key.length(); ++i) {
                    currentChar = key.charAt(i);
                    if (!matchedAlpha && currentChar == ' ') continue;
                    matchedAlpha = true;
                    this.unread(currentChar);
                    if (currentChar != ' ' && currentChar != '\n') continue;
                    StringBuilder newKey = new StringBuilder();
                    for (int j = i; j < key.length(); ++j) {
                        currentChar = key.charAt(j);
                        if (Character.isWhitespace(currentChar)) continue;
                        newKey.append(currentChar);
                    }
                    this.parserResult.addWarning(Localization.lang("Line %0: Found corrupted BibTeX key.", String.valueOf(this.line)));
                    key = newKey.reverse();
                }
                break;
            }
            case ',': {
                this.parserResult.addWarning(Localization.lang("Line %0: Found corrupted BibTeX key (contains whitespaces).", String.valueOf(this.line)));
                break;
            }
            case '\n': {
                this.parserResult.addWarning(Localization.lang("Line %0: Found corrupted BibTeX key (comma missing).", String.valueOf(this.line)));
                break;
            }
            default: {
                this.unreadBuffer(key);
                return "";
            }
        }
        return this.removeWhitespaces(key).toString();
    }

    private StringBuilder removeWhitespaces(StringBuilder toRemove) {
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < toRemove.length(); ++i) {
            char current = toRemove.charAt(i);
            if (Character.isWhitespace(current)) continue;
            result.append(current);
        }
        return result;
    }

    private void unreadBuffer(StringBuilder stringBuilder) throws IOException {
        for (int i = stringBuilder.length() - 1; i >= 0; --i) {
            this.unread(stringBuilder.charAt(i));
        }
    }

    private String parseKey() throws IOException {
        int character;
        StringBuilder token = new StringBuilder(20);
        while (true) {
            if ((character = this.read()) == -1) {
                this.eof = true;
                return token.toString();
            }
            if (Character.isWhitespace((char)character) || !Character.isLetterOrDigit((char)character) && character != 58 && "#{}~,=\ufffd".indexOf(character) != -1) break;
            token.append((char)character);
        }
        if (Character.isWhitespace((char)character)) {
            return token + this.fixKey();
        }
        if (character == 44 || character == 125) {
            this.unread(character);
            return token.toString();
        }
        if (character == 61) {
            return token.toString();
        }
        throw new IOException("Error in line " + this.line + ":Character '" + (char)character + "' is not allowed in bibtex keys.");
    }

    private String parseBracketedText() throws IOException {
        StringBuilder value = new StringBuilder();
        this.consume('{', '(');
        int brackets = 0;
        while (!this.isClosingBracketNext() || brackets != 0) {
            int character = this.read();
            if (this.isEOFCharacter(character)) {
                throw new IOException("Error in line " + this.line + ": EOF in mid-string");
            }
            if (character == 123 || character == 40) {
                ++brackets;
            } else if (character == 125 || character == 41) {
                --brackets;
            }
            if (Character.isWhitespace((char)character)) {
                String whitespacesReduced = this.skipAndRecordWhitespace(character);
                if (!whitespacesReduced.isEmpty() && !"\n\t".equals(whitespacesReduced)) {
                    whitespacesReduced = whitespacesReduced.replace("\t", "");
                    value.append(whitespacesReduced);
                    continue;
                }
                value.append(' ');
                continue;
            }
            value.append((char)character);
        }
        this.consume('}', ')');
        return value.toString();
    }

    private boolean isClosingBracketNext() {
        try {
            int peek = this.peek();
            boolean isCurlyBracket = peek == 125;
            boolean isRoundBracket = peek == 41;
            return isCurlyBracket || isRoundBracket;
        }
        catch (IOException e) {
            return false;
        }
    }

    private StringBuilder parseBracketedTextExactly() throws IOException {
        StringBuilder value = new StringBuilder();
        this.consume('{');
        int brackets = 0;
        char lastCharacter = '\u0000';
        while (true) {
            char character;
            boolean isClosingBracket;
            boolean bl = isClosingBracket = (character = (char)this.read()) == '}' && lastCharacter != '\\';
            if (isClosingBracket && brackets == 0) {
                return value;
            }
            if (this.isEOFCharacter(character)) {
                throw new IOException("Error in line " + this.line + ": EOF in mid-string");
            }
            if (character == '{' && !this.isEscapeSymbol(lastCharacter)) {
                ++brackets;
            } else if (isClosingBracket) {
                --brackets;
            }
            value.append(character);
            lastCharacter = character;
        }
    }

    private boolean isEscapeSymbol(char character) {
        return '\\' == character;
    }

    private StringBuilder parseQuotedFieldExactly() throws IOException {
        StringBuilder value = new StringBuilder();
        this.consume('\"');
        int brackets = 0;
        while (this.peek() != 34 || brackets != 0) {
            int j = this.read();
            if (this.isEOFCharacter(j)) {
                throw new IOException("Error in line " + this.line + ": EOF in mid-string");
            }
            if (j == 123) {
                ++brackets;
            } else if (j == 125) {
                --brackets;
            }
            value.append((char)j);
        }
        this.consume('\"');
        return value;
    }

    private void consume(char expected) throws IOException {
        int character = this.read();
        if (character != expected) {
            throw new IOException("Error in line " + this.line + ": Expected " + expected + " but received " + (char)character);
        }
    }

    private boolean consumeUncritically(char expected) throws IOException {
        int character;
        while ((character = this.read()) != expected && character != -1 && character != 65535) {
        }
        if (this.isEOFCharacter(character)) {
            this.eof = true;
        }
        return character == expected;
    }

    private void consume(char firstOption, char secondOption) throws IOException {
        int character = this.read();
        if (character != firstOption && character != secondOption) {
            throw new IOException("Error in line " + this.line + ": Expected " + firstOption + " or " + secondOption + " but received " + (char)character);
        }
    }
}

